<?php
/**
 * Debug version of register-user.php
 * نسخه Debug برای پیدا کردن خطا
 */

// نمایش تمام خطاها
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');

// Log شروع
error_log("=== Register Debug Started ===");

try {
    // بررسی فایل‌های مورد نیاز
    $configPath = __DIR__ . '/config/config.php';
    $functionsPath = __DIR__ . '/includes/functions.php';
    
    if (!file_exists($configPath)) {
        throw new Exception("❌ فایل config.php پیدا نشد: $configPath");
    }
    
    if (!file_exists($functionsPath)) {
        throw new Exception("❌ فایل functions.php پیدا نشد: $functionsPath");
    }
    
    error_log("✅ فایل‌ها وجود دارند");
    
    require_once $configPath;
    error_log("✅ config.php لود شد");
    
    require_once $functionsPath;
    error_log("✅ functions.php لود شد");
    
    // بررسی توابع مورد نیاز
    $requiredFunctions = [
        'convertPersianToEnglish',
        'validateMobile',
        'getUserByMobile',
        'registerWithOTP',
        'getDBConnection'
    ];
    
    $missingFunctions = [];
    foreach ($requiredFunctions as $func) {
        if (!function_exists($func)) {
            $missingFunctions[] = $func;
        }
    }
    
    if (!empty($missingFunctions)) {
        throw new Exception("❌ توابع زیر وجود ندارند: " . implode(', ', $missingFunctions));
    }
    
    error_log("✅ تمام توابع موجود هستند");
    
    // Only allow POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode([
            'status' => 'error', 
            'message' => 'Method not allowed',
            'debug' => 'فقط POST مجاز است'
        ]);
        exit;
    }
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    $phone = $input['phone'] ?? '';
    
    error_log("📞 شماره دریافت شده: $phone");
    
    // Validate
    $phone = convertPersianToEnglish($phone);
    error_log("📞 شماره تبدیل شده: $phone");
    
    if (!validateMobile($phone)) {
        error_log("❌ شماره نامعتبر است");
        http_response_code(400);
        echo json_encode([
            'status' => 'error', 
            'message' => 'شماره موبایل نامعتبر است',
            'debug' => "Phone: $phone"
        ]);
        exit;
    }
    
    error_log("✅ شماره معتبر است");
    
    // Check if user already exists
    error_log("🔍 بررسی وجود کاربر...");
    $existingUser = getUserByMobile($phone);
    
    if ($existingUser) {
        error_log("⚠️ کاربر وجود دارد: " . json_encode($existingUser));
        http_response_code(400);
        echo json_encode([
            'status' => 'error', 
            'message' => 'این شماره موبایل قبلاً ثبت شده است',
            'debug' => [
                'user_id' => $existingUser['id'] ?? 'N/A',
                'mobile' => $existingUser['mobile'] ?? 'N/A'
            ]
        ]);
        exit;
    }
    
    error_log("✅ کاربر جدید است");
    
    // Register user using the registerWithOTP function
    error_log("📝 شروع ثبت نام...");
    $registerResult = registerWithOTP($phone, null, null, null);
    error_log("📝 نتیجه ثبت نام: " . json_encode($registerResult));
    
    if ($registerResult['success']) {
        error_log("✅ ثبت نام موفق");
        echo json_encode([
            'status' => 'ok',
            'message' => $registerResult['message'],
            'token' => $registerResult['token'] ?? null,
            'user' => $registerResult['user'] ?? null,
            'debug' => 'Success'
        ]);
    } else {
        error_log("❌ ثبت نام ناموفق: " . $registerResult['message']);
        http_response_code(500);
        echo json_encode([
            'status' => 'error', 
            'message' => $registerResult['message'],
            'debug' => $registerResult
        ]);
    }
    
} catch (Exception $e) {
    error_log("💥 Exception: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'خطای سرور',
        'debug' => [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => explode("\n", $e->getTraceAsString())
        ]
    ]);
}

error_log("=== Register Debug Ended ===");
?>

